#include <exec/exec.h>
#include <dos/dos.h>
#include <proto/expansion.h>
#include <proto/exec.h>
#include <proto/dos.h>

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <clib/exec_protos.h>

#include "catweasel.h"
#include "dev_cmd.h"
#include "cwfloppy.h"

UWORD SupportedCommands[]=
{
    CMD_RESET,
    CMD_READ,
    CMD_WRITE,
    CMD_UPDATE,
    CMD_CLEAR,
    CMD_STOP,
    CMD_START,
    CMD_FLUSH,

    TD_MOTOR,
    TD_SEEK,
    TD_FORMAT,
    TD_REMOVE,
    TD_CHANGENUM,
    TD_CHANGESTATE,
    TD_PROTSTATUS,
    TD_ADDCHANGEINT,
    TD_REMCHANGEINT,
    TD_GETGEOMETRY,
    TD_EJECT,
    TD_RAWREAD,
    TD_RAWWRITE,

//  TD_GETNUMTRACKS,
//  TD_GETDRIVETYPE,

//  ETD_READ,
//  ETD_WRITE,
//  ETD_UPDATE,
//  ETD_CLEAR,
//  ETD_MOTOR,
//  ETD_SEEK,
//  ETD_FORMAT,

    NSCMD_DEVICEQUERY,

    0xD001,
    0
};

dev_cmd_type dev_cmd_list[] =
{
    NSCMD_DEVICEQUERY,  (char *)"NSCMD_DEVICEQUERY",    DEV_CMD_NSCMD_DEVICEQUERY,
    CMD_READ,               (char *)"CMD_READ",         read_cyl,
    CMD_WRITE,              (char *)"CMD_WRITE",            write_cyl,
    CMD_UPDATE,             (char *)"CMD_UPDATE",           cmd_update,
    CMD_CLEAR,              (char *)"CMD_CLEAR",            clear_cyl,
    CMD_RESET,              (char *)"CMD_RESET",            DEV_CMD_VOID,

    TD_FORMAT,              (char *)"TD_FORMAT",            format_cyl,
    TD_ADDCHANGEINT,        (char *)"TD_ADDCHANGEINT",      DEV_CMD_ADDCHANGEINT,
    TD_MOTOR,               (char *)"TD_MOTOR",             DEV_CMD_MOTOR,
    TD_CHANGESTATE,         (char *)"TD_CHANGESTATE",       DEV_CMD_TD_CHANGESTATE,
    TD_PROTSTATUS,          (char *)"TD_PROTSTATUS",        DEV_CMD_TD_PROTSTATUS,
    TD_RAWREAD,             (char *)"TD_RAWREAD",           rawread_cyl,
    TD_RAWWRITE,           (char *) "TD_RAWWRITE",          rawwrite_cyl,

    TD_REMCHANGEINT,        (char *)"TD_REMCHANGEINT",      DEV_CMD_VOID,
    TD_REMOVE,              (char *)"TD_REMOVE",            DEV_CMD_VOID,
    TD_SEEK,                (char *)"TD_SEEK",              DEV_CMD_VOID,
    TD_CHANGENUM,           (char *)"TD_CHANGENUM",     DEV_CMD_VOID,
    TD_EJECT,               (char *)"TD_EJECT",             cmd_eject,

//  ETD_READ,               "ETD_READ",             read_cyl,
//  ETD_WRITE,              "ETD_WRITE",                write_cyl,
//  ETD_FORMAT,             "ETD_FORMAT",           format_cyl,
    ETD_MOTOR,              (char *)"ETD_MOTOR",            DEV_CMD_MOTOR,
    ETD_UPDATE,             (char *)"ETD_UPDATE",           cmd_update,
    ETD_SEEK,               (char *)"ETD_SEEK",             DEV_CMD_VOID,
    ETD_CLEAR,              (char *)"ETD_CLEAR",            clear_cyl,

    TD_GETNUMTRACKS,        (char *)"TD_GETNUMTRACKS",      DEV_CMD_GETNUMTRACKS,
    TD_GETDRIVETYPE,        (char *)"TD_GETDRIVETYPE",      DEV_CMD_GETDRIVETYPE,
    TD_GETGEOMETRY,         (char *)"TD_GETGEOMETRY",       DEV_CMD_GETGEOMETRY,
    0xD001,                 (char *)"CNT_CMD_SETFORMAT",    DEV_CMD_SETFORMAT,
    0xD002,                 (char *)"CHK_ENCODE",           check_encode,
    0xD003,                 (char *)"CW_MEMORY_CHECK",  cw4_memory_check,
    0xD004,                 (char *)"EXTERNAL CODE",        ext_code,
    0,                      (char *)"NULL",                 NULL
};



void motor_on(struct cwDevUnit *unit)
{
    if ( unit->motor == 1 )
        return;
    write_txt( unit->driver->debug_port, (char *)"Turning motor on.\n" );
//    unit -> track  = -1;
    unit -> motor = 1;

    if ( unit->motorChanged == 0 )
    {
        catweasel_select(unit -> controller, unit -> cw_unit_num == 0, unit -> cw_unit_num == 1);
        catweasel_set_motor(unit , unit -> motor);
        cw_msdelay(unit -> controller -> ms_timer, 500);
    }

    unit->motorChanged = 0;
}

void motor_off(struct cwDevUnit *unit)
{
    // Seeking is done RELATIVE to the current position, except recalibration. Hence
    // we don't want to lose track of what track we're on, now, do we?
//    unit -> track  = -1;
    write_txt( unit->driver->debug_port, (char *)"Turning motor off.\n" );
    unit -> motor = 0;
    unit->motorChanged = 1;
//    catweasel_select(unit -> controller, unit -> cw_unit_num == 0, unit -> cw_unit_num == 1);
//    catweasel_set_motor(unit , unit -> motor);
//    cw_msdelay(unit -> controller -> ms_timer, 500);
}

int DEV_CMD_MOTOR(struct cwDevUnit *unit,struct IOExtTD *ioreq)
{
//    unit -> track = -1;
    write_num( unit->driver->debug_port, (char *)"CMD_MOTOR: Motor is", unit->motor );
    ioreq->iotd_Req.io_Actual = unit->motor;
    if ( ioreq->iotd_Req.io_Length )
        motor_on( unit );
    else
        motor_off( unit );
//    catweasel_select(unit -> controller, unit -> cw_unit_num == 0, unit -> cw_unit_num == 1);
//    catweasel_set_motor(unit , unit -> motor);
//    cw_msdelay(unit -> controller -> ms_timer, 500);
    write_num( unit->driver->debug_port, (char *)"Motor is now", unit->motor );
    write_num( unit->driver->debug_port, (char *)"motor length set to", ioreq->iotd_Req.io_Length );
    write_num( unit->driver->debug_port, (char *)"motor was", ioreq->iotd_Req.io_Actual );
    ioreq->iotd_Req.io_Error = 0;
    return 0;
}

int DEV_CMD_SETFORMAT(struct cwDevUnit *unit,struct IOExtTD *ioreq)
{
    unit -> format = ioreq -> iotd_Req.io_Length;

    unit -> drive_geo ->dg_SectorSize       =   cw_formats[ unit -> format ].blksize;
    unit -> drive_geo ->dg_Cylinders        =   cw_formats[ unit -> format ].cylinders;
    unit -> drive_geo ->dg_Heads        =   cw_formats[ unit -> format ].heads;
    unit -> drive_geo ->dg_TrackSectors =   cw_formats[ unit -> format ].max_secs_per_track;

    unit -> drive_geo ->dg_CylSectors   =   unit -> drive_geo ->dg_TrackSectors * unit -> drive_geo ->dg_Heads ;
    unit -> drive_geo ->dg_TotalSectors =   unit -> drive_geo ->dg_Cylinders * unit -> drive_geo ->dg_CylSectors;
    unit -> drive_geo ->dg_Flags        = DGB_REMOVABLE;

    return 0;
}



int clear_track(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    int             tracksize;
    int             index;
    int             fd;

    if (catweasel_write_protected(unit))
        return TDERR_WriteProt;

    fd          = unit -> driver -> fd;
    tracksize       = cw_formats[ unit -> format ].blksize * cw_formats[ unit -> format ].max_secs_per_track;

    index       = (ioreq -> iotd_Req.io_Offset / tracksize);

    if( unit->buffered[ index ] ) 
    {
#if defined(AROS)
#else
        IExec -> 
#endif        
        FreeVec(  unit->buffered[ index ] );
        unit->buffered[ index ] = NULL;
    }
    ioreq ->  iotd_Req.io_Actual = ioreq ->  iotd_Req.io_Length;
    return 0;
}

void cw_reset_ram_pointer_w(struct cwDevUnit *unit)
{
    struct PCIDevice    *pcicard = unit -> controller -> pcicard;

    if(unit -> controller -> type == CATWEASEL_TYPE_MK1)
    {
        PCICARD_INBYTE( pcicard, unit->controller->io_base + 1);
    } else {
        PCICARD_OUTBYTE( pcicard, unit->controller->io_base + 0xe4,0); // Reset memory pointer 
    }
}

void cw_reset_ram_pointer_r(struct cwDevUnit *unit)
{
    struct PCIDevice    *pcicard = unit -> controller -> pcicard;

    if(unit -> controller -> type == CATWEASEL_TYPE_MK1) 
    {
        PCICARD_INBYTE( pcicard, unit -> controller -> io_base + 1);
        PCICARD_INBYTE( pcicard, unit -> controller -> io_mem); /* ignore first byte */
    } 
    else 
    {
        PCICARD_OUTBYTE( pcicard, unit -> controller -> io_base + 0xe4,0);
    }
}


int read_track (struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    int             s = 0;
    int             i = 0;
    int             tryNumber;
    struct MsgPort  *port;
    struct          cw_driver *driver;
    struct PCIDevice    *pcicard = unit -> controller -> pcicard;
    int             index;
    int             track;
    int             tracksize;
    int             error = 0;
    char                *data = (char *)ioreq ->  iotd_Req.io_Data;
    int             motor_forced_on = 0;

    ioreq ->  iotd_Req.io_Actual =0;

    if (unit -> motor == 0)
    {
        motor_forced_on=1 ;
        motor_on(unit);
    }

    tracksize = cw_formats[ unit -> format ].blksize * cw_formats[ unit -> format ].max_secs_per_track;

    index       = ioreq -> iotd_Req.io_Offset / tracksize ;
    track       = index / 2;
    unit    -> side = index % 2;
    driver      = unit -> driver;
    port            = unit->driver->debug_port;

    write_num_no_enter(port, (char *)"Reading Track", track);
    write_num_no_enter(port,(char *)"  Side", index % 2);
    write_num(port,(char *)"  Sector",( ioreq -> iotd_Req.io_Offset % tracksize ) / cw_formats[ unit -> format ].blksize);

    if( unit->buffered[ index ] ) 
    {
        write_txt( port, "Using cached data.\n" );
        unit -> trackbuffer = unit -> buffered[ index ];
        goto ok;
    }

    write_txt( port, "Reading data from disk...\n" );

    unit -> trackbuffer =
    
#if defined(AROS)
        (unsigned char *)AllocVec( tracksize, MEMF_CLEAR );
#else   
        IExec -> AllocVec( tracksize , MEMF_CLEAR | MEMF_PRIVATE );
#endif
    if (!unit -> trackbuffer)
        return 0;

    catweasel_select(unit -> controller, unit -> cw_unit_num == 0, unit -> cw_unit_num == 1);

    if(!catweasel_seek(unit, track)) 
    {
        error = TDERR_SeekError;
        goto err;
    }

    if (unit -> track != track)
    {
        error = TDERR_SeekError;
        goto err;
    }

    for(tryNumber = 0; tryNumber < 10; tryNumber++) 
    {
        cw_reset_ram_pointer_r(unit);
        if(!catweasel_read(unit))
        {
            error = TDERR_DiskChanged;
            s = 0;
            break;
        }

        cw_reset_ram_pointer_r(unit);
        //write_num( port, "Side is ", unit->side );
         write_num( port, (char *)"Motor is ", unit->motor );
        s = cw_formats[unit->format].codec->decode(unit, tryNumber);

        if(s)
        {
            error = 0;
            break;
        }
        else 
        {
            error = TDERR_TooFewSecs ;

            if(tryNumber > 2) 
            {
#ifdef debug
                write_txt( unit -> driver -> debug_port , (char *)"Recalibrate\n");
#endif
                unit -> track = -1;

                /* recalibrate */
                if(!catweasel_seek(unit, 0)) 
                {
                    error = TDERR_SeekError;
                    break;
                }

                if(!catweasel_seek(unit, track))
                {
                    error = TDERR_SeekError;
                    break;
                }
            }
        }
    }
err:
    catweasel_select( unit->controller , 0, 0);
ok:
    if (!error)
    {
        //write_txt( port, "Read: Setting buffered[] to trackbuffer.\n" );
        unit->buffered[ index ] = unit->trackbuffer;

        // store changes to the io req
        for ( i=0 ; i<ioreq->iotd_Req.io_Length ; i++ )
            data[ i ] = unit->trackbuffer[i + ( ioreq -> iotd_Req.io_Offset % tracksize ) ];

        ioreq -> iotd_Req.io_Actual = ioreq -> iotd_Req.io_Length;
    }
out:
    if (error)
    {
        if (unit->trackbuffer)
        {
#if defined(AROS)
#else
            IExec->
#endif            
            FreeVec(unit -> trackbuffer);
            unit->trackbuffer = NULL;
            write_txt( port, (char *)"Read: Zeroing buffered[]" );
            unit->buffered[ index ] = NULL;
        }

        unit -> track = -1;
    }

    if ( motor_forced_on == 1 )
        motor_off( unit );

    return error;
}

int rawread_track (struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    int             s = 0;
    int             i = 0;
    int             tryNumber;
    struct MsgPort  *port;
    struct          cw_driver *driver;
    struct PCIDevice    *pcicard = unit -> controller -> pcicard;
    int             index;
    int             track;
    int             tracksize;
    int             error = 0;
    char                *data = (char *)ioreq ->  iotd_Req.io_Data;
    int             motor_forced_on = 0;
    char            *tRawData = NULL;
    int             tBytesRead;

    ioreq ->  iotd_Req.io_Actual =0;

    if (unit -> motor == 0)
    {
        motor_forced_on=1 ;
        motor_on(unit);
    }

    tracksize = cw_formats[ unit -> format ].blksize * cw_formats[ unit -> format ].max_secs_per_track;

    index       = ioreq -> iotd_Req.io_Offset;
    track       = index / 2;
    unit    -> side = index % 2;
    driver      = unit -> driver;
    port            = unit->driver->debug_port;

    write_num_no_enter(port, (char *)"Raw Reading Track", track);
    write_num_no_enter(port,(char *)"  Side", index % 2);

#if defined(AROS)
    tRawData = (char * )AllocVec( ioreq->iotd_Req.io_Length, MEMF_CLEAR );
#else
    tRawData = IExec -> AllocVec( ioreq->iotd_Req.io_Length, MEMF_CLEAR | MEMF_PRIVATE );
#endif
    if ( tRawData == NULL )
        return 0;

    catweasel_select(unit -> controller, unit -> cw_unit_num == 0, unit -> cw_unit_num == 1);

    if(!catweasel_seek(unit, track))
    {
        error = TDERR_SeekError;
        goto err;
    }

    if (unit -> track != track)
    {
        error = TDERR_SeekError;
        goto err;
    }

    cw_reset_ram_pointer_r(unit);       
    
    tBytesRead = 0;
    tBytesRead = catweasel_read_raw_data( unit, (BYTE *)tRawData, ioreq->iotd_Req.io_Length );
    if ( tBytesRead > 0 )
    {
        // We've read some bytes!
        write_hex( port,(char *) "Read Raw MFM data bytes:", tBytesRead );

#if defined(AROS)
#else
        IExec->
#endif        
        CopyMem( tRawData, ioreq->iotd_Req.io_Data, tBytesRead );
        write_hex( port, (char *)"Copied into", (int)ioreq->iotd_Req.io_Data );
    }
    else
        write_txt( port, (char *)"Failed to read any raw MFM data." );

#if defined(AROS)
#else
    IExec->
#endif    
    FreeVec( tRawData );

err:
    catweasel_select( unit->controller , 0, 0);
    ioreq -> iotd_Req.io_Actual = tBytesRead;

    if ( motor_forced_on == 1 )
        motor_off( unit );

    return error;
}



int check_encode(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    int             i;
    int             tracksize;
    char                *data;
    struct PCIDevice    *pcicard = unit -> controller -> pcicard;

    tracksize   = cw_formats[ unit -> format ].blksize * cw_formats[ unit -> format ].max_secs_per_track;
    data        = (char *)ioreq ->  iotd_Req.io_Data;

    if (unit -> trackbuffer = 
#if defined(AROS)    
    (unsigned char *)AllocVec( tracksize , MEMF_CLEAR  ))
#else
    IExec -> AllocVec( tracksize , MEMF_CLEAR | MEMF_PRIVATE )) 
#endif    
    {
        catweasel_select(unit -> controller, unit -> cw_unit_num == 0, unit -> cw_unit_num == 1);

        for ( i=0 ; i<ioreq->iotd_Req.io_Length ; i++ )    
            unit->trackbuffer[ i + (ioreq->iotd_Req.io_Offset % tracksize) ] = data[ i ];

        // CW3 / CW4
        CWSetCReg(unit->controller, 1<<6, (!unit->side)<<6); // set disk side 
        PCICARD_OUTBYTE( pcicard, unit->controller->io_base + 0xe4,0); // Reset memory pointer 

        // encode data from buffer
        cw_formats[unit->format].codec -> encode(unit);

        // zero data
        // for (i=0;i<ioreq ->  iotd_Req.io_Length;i++)     { unit -> trackbuffer[i + ( ioreq -> iotd_Req.io_Offset % tracksize) ] = 0; }

        // CW3 / CW4
        CWSetCReg(unit->controller, 1<<6, (!unit->side)<<6); // set disk side 
        PCICARD_OUTBYTE( pcicard, unit->controller->io_base + 0xe4,0); // Reset memory pointer 

        // decode data from cw
        cw_formats[unit->format].codec -> decode(unit, 1);

        catweasel_select(unit->controller, 0, 0);

        // copy data back
        for (i=0;i<ioreq ->  iotd_Req.io_Length;i++)    { data[i] = unit -> trackbuffer[i + ( ioreq -> iotd_Req.io_Offset % tracksize) ]; }

#if defined(AROS)
#else
        IExec -> 
#endif        
        FreeVec( unit -> trackbuffer  );
    }

    return 0;
}

int write_track(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    struct cw_driver    *driver = NULL;
    struct PCIDevice    *pcicard = unit -> controller -> pcicard;
    struct MsgPort  *port;
    char                *data ;
    int             tracksize;
    int             track;
    int             index;
    int             error = 0;
    int             retry = 0;
    int             i,ii;
    int             crc_error = 0;
    int             cnt_write = 0;
    int             maxsize;
    int             length;
    int             sector = 0;
    char            *tOldData = NULL;
    int             tActualLength = 0;
    int             motor_forced_on = 0;
    int offset;

    if (catweasel_write_protected(unit))
        return TDERR_WriteProt;

    tracksize       = cw_formats[ unit -> format ].blksize * cw_formats[ unit -> format ].max_secs_per_track;

    offset = ioreq->iotd_Req.io_Offset;

    if (unit -> motor == 0)
    {
        motor_forced_on=1 ;
        motor_on(unit);
    }

    data            = (char *)ioreq ->  iotd_Req.io_Data;
    index       = ioreq -> iotd_Req.io_Offset / tracksize ;
    track       = index / 2;
    sector = ( ioreq -> iotd_Req.io_Offset % tracksize ) / cw_formats[ unit -> format ].blksize;
    unit    -> side = index % 2;

    port            = unit->driver->debug_port;
    driver      = unit -> driver;

    write_num_no_enter(port,(char *)"Writing Track", index / 2);
    write_num_no_enter(port,(char *)"  Side", index % 2);
    write_num(port,(char *)"  Sector", sector);

    if( !unit->buffered[ index ] ) 
    {
        write_txt(port,(char *)"** WARNING! NO DATA TO OVER WRITE **\n");

        // do not allow read to overwrite iodata

        length = ioreq ->  iotd_Req.io_Length;
        tActualLength = ioreq->iotd_Req.io_Actual;

#if defined(AROS)
        tOldData = (char *)AllocVec( tracksize, MEMF_CLEAR );
#else
        tOldData = IExec->AllocVec( tracksize, MEMF_PRIVATE | MEMF_CLEAR );
#endif
        ioreq ->  iotd_Req.io_Length = tracksize;
        ioreq ->  iotd_Req.io_Data = tOldData;
        ioreq->iotd_Req.io_Offset -= offset % tracksize;

        write_hex( port, (char *)"offset was", offset );
        write_hex( port, (char *)"offset is now", ioreq->iotd_Req.io_Offset );
        write_hex( port, (char *)"tracksize is", tracksize );

//        if ( sector == 0 && length == tracksize )
//            write_txt( port, " Overwriting entire track.\n" );
//        else
            read_track(unit,ioreq);

        // if we can't get any data we force it
        if( !unit->buffered[ index ] )
        {
#if defined(AROS)
            unit->buffered[ index ]  = (unsigned char *) AllocVec(tracksize, MEMF_CLEAR );
#else        
            unit->buffered[ index ]  = (void *) IExec -> AllocVec(tracksize,MEMF_PRIVATE | MEMF_CLEAR );
#endif
            write_txt(port,(char *)"**** DISK ERROR: Ignore data on disk ****\n");
        }

        // Reset io length so write can do it's work.
        ioreq->iotd_Req.io_Length = length;
        ioreq->iotd_Req.io_Data = data;
        ioreq->iotd_Req.io_Offset = offset;
        ioreq->iotd_Req.io_Actual = tActualLength;
    }

    if ( (unit->trackbuffer = unit->buffered[ index ]) != NULL )
    {
        write_txt( port, (char *)"updating buffer." );
        // Copy the old data into the buffer....
        if ( tOldData )
            for ( i=0 ; i<tracksize ; i++ )
                unit->trackbuffer[ i ] = tOldData[ i ];

        write_txt( port, (char *)"copying in new data." );

        // And over-write it with lovely new data.
        for ( i=0 ; i<ioreq->iotd_Req.io_Length ; i++ )
            unit->trackbuffer[ i + ( ioreq->iotd_Req.io_Offset % tracksize ) ] = data[ i ];

//        char tBuffer[ 1024 ] ;
//        sprintf( tBuffer, "Written data to buffered[ %d ]. Value is 0x%02x%02x%02x%02x\n", index, unit->trackbuffer[ 0 ], unit->trackbuffer[ 1 ], unit->trackbuffer[ 2 ], unit->trackbuffer[ 3 ] );
//        write_txt( port, tBuffer );

        unit->update[ index ] = 1;
        unit->trackbuffer = 0;
    }
    else
    {
        unit->update[ index ] = 0;
        error = TDERR_TooFewSecs;
    }

    if ( tOldData )
#if defined(AROS)
#else
        IExec->
#endif        
        FreeVec( tOldData );

    if ( error )
        write_hex( port, (char *)"Write: Error ", error );

    if ( motor_forced_on == 1 )
        motor_off( unit );


    return error;
//    REL_HW;
}

int rawwrite_track(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    struct cw_driver    *driver = NULL;
    struct PCIDevice    *pcicard = unit -> controller -> pcicard;
    char                *data ;
    struct MsgPort  *port;
    int             tracksize;
    int             track;
    int             index;
    int             error = 0;
    int             retry = 0;
    int             i,ii;
    int             crc_error = 0;
    int             cnt_write = 0;
    int             maxsize;

    tracksize       = ioreq->iotd_Req.io_Length;;
    data            = (char *)ioreq->iotd_Req.io_Data;
    track           = (ioreq->iotd_Req.io_Offset)/2;
    unit->side      = ioreq->iotd_Req.io_Offset%2;

    port            = unit -> driver -> debug_port;
    driver          = unit -> driver;

    index = track * 2 + unit -> side ;

    catweasel_select(unit -> controller, unit -> cw_unit_num == 0, unit -> cw_unit_num == 1);

    retry = 3;
    do
    {
        catweasel_seek(unit, track );
        if (unit -> track != track)
        {   // ERROR
             error = TDERR_SeekError;
            retry--;
        }
        else
        {   // OK
            error = 0;
            retry = 0;
        }
    } while ((error>0)&&(retry>0));
    if (error)
        goto err;

    CWSetCReg(unit->controller, 1<<6, (!unit->side)<<6); // set disk side
    PCICARD_OUTBYTE( pcicard, unit->controller->io_base + 0xe4,0); // Reset memory pointer

    PCICARD_OUTBYTE( pcicard, unit->controller->io_mem,0xff);

    write_num( port, (char *)"Raw Writing bytes:", tracksize );
    catweasel_write_raw_data( unit, (BYTE *)data, tracksize );
    catweasel_write(unit);

err:
    catweasel_select(unit->controller, 0, 0);

    return error;
//    REL_HW;
}

int cmd_update(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    struct PCIDevice    *pcicard = unit -> controller -> pcicard;
    struct MsgPort  *port;
    int i,a,notvalid=1;
    int track;
    int error = 0;
    int retry = 0;
    int tracksize = 0;
    int motor_forced_on = 0;

    port = unit->driver->debug_port;

    catweasel_select(unit -> controller, unit -> cw_unit_num == 0, unit -> cw_unit_num == 1);

    write_txt( port, (char *)"Motor on.." );

    if (unit -> motor == 0)
    {
        motor_forced_on=1 ;
        motor_on(unit);
    }

    write_txt( port, (char *)"OK." );
    write_hex( port, (char *)"unit is", (int)unit );
    write_hex( port, (char *)"format is", unit->format );

    tracksize = cw_formats[ unit -> format ].blksize * cw_formats[ unit -> format ].max_secs_per_track;

    for(i=0;i<200;i++)
    {
        if (unit -> update[i])
        {
            track = i / 2;
            unit -> side = i % 2;

#if defined(AROS)
            if (unit -> trackbuffer = (unsigned char *)AllocVec( tracksize , MEMF_CLEAR) )
#else
            if (unit -> trackbuffer = IExec -> AllocVec( tracksize , MEMF_PRIVATE | MEMF_CLEAR) )
#endif
            {
                write_num_no_enter(port,(char *)"  Update track",track);
                write_num(port,(char *)"  Side",unit -> side);
//                write_num(port,"**  IO REQ Length",ioreq ->  iotd_Req.io_Length);

                retry = 3;
//                write_txt( port, "seeking to zero..." );
//                if ( catweasel_seek( unit, 0 ) == 0 )
//                    write_txt( port, "failed!" );
                do
                {
                    catweasel_seek(unit, track );
                    if (unit -> track != track) 
                    { 
                        error = TDERR_SeekError; 
                        retry--;
                    }
                    else
                    { 
                        error = 0;
                        retry = 0;  
                    } 
                } while ( error > 0 && retry > 0 );

                if (error) 
                {
                    write_txt(port,(char *)"** SEEK FAILED **\n");
                    goto err;
                }

                retry = 4;

                do
                {
                    write_hex( port, (char *)"Writing data from ", (int) unit->buffered[ i ] );
                    write_hex( port, (char *)"Data is 0x", unit->buffered[ i ][ 0 ] );
                    // Clone data
                    for (a=0;a<tracksize;a++)
                        unit->trackbuffer[ a ] = unit->buffered[ i ][ a ];

#if defined(AROS)
#else
                    unit->driver->I_Exec->
#endif                    
                    Forbid();

                    cw_reset_ram_pointer_w( unit );
                    write_num( port, (char *)"Encoding track", unit->track );
                    cw_formats[ unit->format ].codec->encode( unit );
                    PCICARD_OUTBYTE( pcicard,  unit->controller->io_mem, 0xff );
                    catweasel_write(unit);

#if defined(AROS)
#else
                    unit->driver->I_Exec->
#endif                    
                    Permit();

                    notvalid = 0;

#if 0
                    // Clear
                    for (a=0;a<tracksize;a++)
                        unit -> trackbuffer[a] = 0;

                    cw_reset_ram_pointer_r(unit);
                    PCICARD_OUTBYTE( pcicard,  unit->controller->io_mem, 0xff );
                    catweasel_read(unit);

                    cw_reset_ram_pointer_r(unit);
                    cw_formats[unit->format].codec -> decode(unit,9);

                    notvalid = 0;
                    write_txt( port, "Checking data...\n" );
                    for (a=0;a<tracksize;a++)
                    {
                        if ( unit->trackbuffer[ a ] != unit->buffered[ i ][ a ] )
                        {
                            write_txt( port, "*********** NOT VALID DATA **************");
                            notvalid = 1;
                            break;
                        }
                    }

                    if (!notvalid)
                    {
                        break;
                    }

                    retry --;
#endif
                } while ( notvalid && retry>0 );

                if (retry==0)
                    write_txt( port, (char *)"***********[( CAN'T WRITE DATA? )]**************");


#if defined(AROS)
#else
                IExec -> 
#endif                
                FreeVec(unit -> trackbuffer);
                unit -> trackbuffer = NULL;
                
                if ( retry == 0 || notvalid )
                {
                    write_txt( port, (char *)"Update: Zeroing buffer" );
#if defined(AROS)
#else
                    IExec->
#endif                    
                    FreeVec(unit -> buffered[ i ]);
                    unit->buffered[ i ] = NULL;
                }
                write_txt(port,(char *)"Done\n");
            }
            else
            {
                error = TDERR_NoMem;
            }
        }

        if (error)
        {
            write_txt(port,(char *)"Error\n");
        }

        unit->update[ i ] = 0; // we don't want a lockup due to one error, so we give up too easy...
    }

err:
    if (motor_forced_on)
        motor_off(unit);

    catweasel_select(unit->controller, 0, 0);
    return error;
}

int format_track(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    struct cw_driver    *driver = NULL;
    struct PCIDevice    *pcicard = unit -> controller -> pcicard;
    char                *data ;
    struct MsgPort  *port;
    int             tracksize;
    int             track;
    int             index;
    int             error = 0;
    int             retry = 0;
    int             i,ii;
    int             crc_error = 0;
    int             cnt_write = 0;
    int             maxsize;

    if (catweasel_write_protected(unit))
        return TDERR_WriteProt;

    tracksize       = cw_formats[ unit -> format ].blksize * cw_formats[ unit -> format ].max_secs_per_track;

    data            = (char *)ioreq ->  iotd_Req.io_Data;
    track       = (ioreq -> iotd_Req.io_Offset / tracksize)/2;
    unit    -> side = (ioreq -> iotd_Req.io_Offset / tracksize)%2;

    port            = unit -> driver -> debug_port;
    driver      = unit -> driver;

    index = track * 2 + unit -> side ;

#ifdef debug
    write_num(port,(char *)"Offset sector nr",ioreq -> iotd_Req.io_Offset / cw_formats[ unit -> format ].blksize);
    write_num(port,(char *)"Sectors to write",ioreq ->  iotd_Req.io_Length / cw_formats[ unit -> format ].blksize);
#endif

    if( unit->buffered[ index ] ) 
    {
        // Use existing...
        unit -> trackbuffer = unit -> buffered[ index ];
    }
    else
    {
        // No track buffer
        unit -> trackbuffer = 
#if defined(AROS)  
        (unsigned char *)AllocVec( tracksize , MEMF_CLEAR );      
#else
        IExec -> AllocVec( tracksize , MEMF_CLEAR | MEMF_PRIVATE ); 
#endif        
        
        unit -> buffered[ index ] = unit -> trackbuffer;

        // Can't get buffer, then it's like it's write protected :-)
        if (!unit -> trackbuffer)
        {
            write_txt( port, (char *)"Can't get a buffer! Returning as write protected." );
            return TDERR_WriteProt;
        }
    }

    for (i=0;i<ioreq ->  iotd_Req.io_Length;i++)
    {
        unit -> trackbuffer[ i + ( ioreq -> iotd_Req.io_Offset % tracksize) ] = data[i];
    }

//    GET_HW;
    catweasel_select(unit -> controller, unit -> cw_unit_num == 0, unit -> cw_unit_num == 1);

    retry = 3;
    do
    {
        catweasel_seek(unit, track );
        if (unit -> track != track)  
        {   // ERROR
             error = TDERR_SeekError; 
            retry--;
        }
        else
        {   // OK
            error = 0;
            retry = 0; 
        } 
    } while ((error>0)&&(retry>0));
    if (error) goto err;

    CWSetCReg(unit->controller, 1<<6, (!unit->side)<<6); // set disk side
    PCICARD_OUTBYTE( pcicard, unit->controller->io_base + 0xe4,0); // Reset memory pointer

    cw_formats[unit->format].codec -> encode(unit);
    PCICARD_OUTBYTE( pcicard, unit->controller->io_mem,0xff);

    catweasel_write(unit);
err:
    catweasel_select(unit->controller, 0, 0);

    return error;
//    REL_HW;
}

int std_cyl(struct cwDevUnit *unit, struct IOExtTD *ioreq,int dev_size,int  (* track_func) (struct cwDevUnit *unit, struct IOExtTD *ioreq) )
{
    struct cw_driver    *driver = NULL;
    struct PCIDevice    *pcicard = unit -> controller -> pcicard;
    char                *data ;
    int             i,ii;
    int             tracksize;
    int             error = 0;
    int             cnt_write = 1;
    int             maxsize;
    int             offset;
    struct IOExtTD  local_ioreq;
    char                *tmp_trackbuffer;

    data        = (char *)ioreq ->  iotd_Req.io_Data;

    maxsize = ioreq ->  iotd_Req.io_Length;
    if (maxsize > dev_size)
    {
        maxsize = dev_size;
        cnt_write = ioreq ->  iotd_Req.io_Length / dev_size;
    }
    ioreq -> iotd_Req.io_Actual = 0;

    write_hex_no_enter(unit->driver->debug_port,(char *)"  index", ioreq->iotd_Req.io_Offset );
    write_hex(unit->driver->debug_port,(char *)"length", ioreq->iotd_Req.io_Length);

    if ( ioreq->iotd_Req.io_Command == TD_RAWREAD || ioreq->iotd_Req.io_Command == TD_RAWWRITE )
    {
        local_ioreq.iotd_Req.io_Offset = 0;
        local_ioreq.iotd_Req.io_Data = (void *) ((char *) ioreq ->  iotd_Req.io_Data );
        local_ioreq.iotd_Req.io_Length = ioreq->iotd_Req.io_Length;
        local_ioreq.iotd_Req.io_Actual = 0;

        error = track_func(unit,&local_ioreq);
        write_txt( unit->driver->debug_port, (char *)"Done raw read/write" );
        ioreq -> iotd_Req.io_Actual +=local_ioreq.iotd_Req.io_Actual;
        write_hex( unit->driver->debug_port, (char *)"Actual bytes", ioreq->iotd_Req.io_Actual );

    }
    else
    {
        offset = 0;
        for (i=0;i<cnt_write;i++)
        {
            local_ioreq.iotd_Req.io_Offset = ioreq -> iotd_Req.io_Offset + offset;
            local_ioreq.iotd_Req.io_Data = (void *) (((char *) ioreq ->  iotd_Req.io_Data) + offset);
            local_ioreq.iotd_Req.io_Length = maxsize;
            local_ioreq.iotd_Req.io_Actual = 0;

            if (local_ioreq.iotd_Req.io_Offset % dev_size)
                error = IOERR_BADADDRESS;
            if (local_ioreq.iotd_Req.io_Length > dev_size)
                error = IOERR_BADLENGTH;

            if (!error)
            {
                error = track_func(unit,&local_ioreq);
                write_txt( unit->driver->debug_port, (char *)"Done read/write" );
                ioreq -> iotd_Req.io_Actual +=local_ioreq.iotd_Req.io_Actual;
                write_hex( unit->driver->debug_port, (char *)"Actual bytes", ioreq->iotd_Req.io_Actual );
            }

            if (error)
                break;

            offset += dev_size;
        }
    }
    if ( error )
        write_hex( unit->driver->debug_port, (char *)"Error", error );

    return error;
}

// write/read whit block aligment

int read_cyl(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    if (unit -> motor == 0)
    {
        write_txt(unit -> driver -> debug_port,(char *)"MOTOR NOT ON, FORCED ON!!!!\n");
        motor_on(unit);
    }

    return std_cyl(unit,ioreq,cw_formats[ unit -> format ].blksize,read_track) ;
}

int rawread_cyl(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    int tMotorForcedOn = 0;
    
    if (unit -> motor == 0)
    {
        write_txt(unit -> driver -> debug_port,(char *)"MOTOR NOT ON, FORCED ON!!!!\n");
        motor_on(unit);
        tMotorForcedOn = 1;
    }

    if ( tMotorForcedOn )
        motor_off( unit );

    return std_cyl(unit,ioreq,cw_formats[ unit -> format ].blksize,rawread_track) ;
}

int write_cyl(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    return std_cyl(unit,ioreq,cw_formats[ unit -> format ].blksize,write_track) ;
}

int rawwrite_cyl(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    return std_cyl(unit,ioreq,cw_formats[ unit -> format ].blksize,rawwrite_track) ;
}


int format_cyl(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    int tracksize = cw_formats[ unit -> format ].blksize * cw_formats[ unit -> format ].max_secs_per_track;
    return std_cyl(unit,ioreq,tracksize,format_track) ;
}

int clear_cyl(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    int tracksize = cw_formats[ unit -> format ].blksize * cw_formats[ unit -> format ].max_secs_per_track;
    return std_cyl(unit,ioreq,tracksize,clear_track) ;
}



int DEV_CMD_TD_CHANGESTATE(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    struct PCIDevice    *pcicard = unit -> controller -> pcicard;
    int             fd = unit -> driver -> fd;
    int             error = 0;

    ioreq ->  iotd_Req.io_Error = error;

    catweasel_select(unit -> controller, unit -> cw_unit_num == 0, unit -> cw_unit_num == 1);

    if(!catweasel_seek(unit, 0)) error = TDERR_SeekError;

    // really this all I need :-), but I'm using the catweasel generic code to be on the safe side.
    // PCICARD_INBYTE( pcicard, unit->controller->io_sr) & unit->controller->srm_dchg);

    catweasel_disk_changed(unit);
    ioreq ->  iotd_Req.io_Actual = 1 - unit->diskindrive;

    return error;
}

int DEV_CMD_TD_PROTSTATUS(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    struct PCIDevice    *pcicard = unit -> controller -> pcicard;
    int             fd = unit -> driver -> fd;
    int             error = 0;

    ioreq ->  iotd_Req.io_Error = error;

    catweasel_select(unit -> controller, unit -> cw_unit_num == 0, unit -> cw_unit_num == 1);

    ioreq -> iotd_Req.io_Actual = catweasel_write_protected(unit);

    return error;
}


int DEV_CMD_NSCMD_DEVICEQUERY(struct cwDevUnit *unit,struct IOExtTD *ioreq)
{
    int                     i;
    struct NSDeviceQueryResult  *edit;
    
    ioreq ->  iotd_Req.io_Actual = 0;

    if ( ioreq -> iotd_Req.io_Length >= sizeof(struct NSDeviceQueryResult) )
    {
        if (edit = (struct NSDeviceQueryResult *) ioreq ->  iotd_Req.io_Data)
        {
            edit -> DevQueryFormat  = 0;
            edit -> SizeAvailable       = 16;
            edit -> DeviceType          = NSDEVTYPE_TRACKDISK;
            edit -> DeviceSubType       = 0;
            edit -> SupportedCommands = &SupportedCommands[0];

            ioreq -> iotd_Req.io_Actual = sizeof(struct NSDeviceQueryResult);
        }
    }
    else
    {
        return IOERR_BADLENGTH;
    }

    write_txt(unit -> driver -> debug_port,(char *)"END ** NSCMD_DEVICEQUERY\n");
    return 0;
}



int ext_code(struct cwDevUnit *unit, struct IOExtTD *ioreq)
{
    int                 i;
    struct ext_code_type    *do_code;
    
    ioreq ->  iotd_Req.io_Actual = 0;

    if ( ioreq -> iotd_Req.io_Length >= sizeof(struct ext_code_type) )
    {
        if (do_code = (struct ext_code_type *) ioreq ->  iotd_Req.io_Data)
        {
            do_code -> code(unit -> controller,do_code -> arg1);

            ioreq -> iotd_Req.io_Actual = sizeof(struct ext_code_type);
        }
    }
}

int DEV_CMD_GETGEOMETRY(struct cwDevUnit *unit,struct IOExtTD *ioreq)
{
    int                     i;
    char                        *data = (char *)ioreq ->  iotd_Req.io_Data;
    char                        *buffer = (char *) unit -> drive_geo;

    if ( ioreq->iotd_Req.io_Length < 16 )
        return IOERR_BADLENGTH;

    for ( i=0 ; (( i<ioreq->iotd_Req.io_Length ) && ( i<sizeof(struct DriveGeometry)) ) ; i++ )
         data[i] = buffer[i];
    ioreq->iotd_Req.io_Actual = i-1;


    return 0;
}

/*
struct Interrupt
{
    struct Node is_Node;
    APTR        is_Data;    // server data segment
    VOID      (*is_Code)(); // server code entry
};
*/

int DEV_CMD_ADDCHANGEINT(struct cwDevUnit *unit,struct IOExtTD *ioreq)
{
    struct MsgPort *port = unit -> driver -> debug_port;
    struct Interrupt *inter;

    unit -> interrupt = (struct Interrupt *) ioreq -> iotd_Req.io_Data;

    return 0;
}

int DEV_CMD_GETDRIVETYPE(struct cwDevUnit *unit,struct IOExtTD *ioreq)
{
    ioreq ->  iotd_Req.io_Actual = DRIVE3_5;
    return 0;
}

int DEV_CMD_GETNUMTRACKS(struct cwDevUnit *unit,struct IOExtTD *ioreq)
{
    ioreq ->  iotd_Req.io_Actual = 80;
    return 0;
}

int DEV_CMD_VOID(struct cwDevUnit *unit,struct IOExtTD *ioreq)
{
    ioreq ->  iotd_Req.io_Actual = 0;
    return 0;
}

int DEV_CMD_PROTSTATUS(struct cwDevUnit *unit,struct IOExtTD *ioreq)
{
    write_txt(unit -> driver -> debug_port,(char *)"CMD_PROTSTATUS\n");
    ioreq ->  iotd_Req.io_Actual = 0;
    return 0;
}

int cmd_eject( struct cwDevUnit *unit, struct IOExtTD *ioreq )
{
    int i,func,error;
    struct IOExtTD  *disk = (struct IOExtTD *) ioreq;
    struct Task     *task;
    struct MsgPort  *port;
    struct catweasel_contr *tController;

    tController = unit->controller;
    port = unit->driver->debug_port;

    int tUnit;
    for ( tUnit = (unit->cw_unit_num % 2) ; tUnit < (2 * NUMBER_OF_SUPPORTED_FORMATS) ; tUnit += 2 )
        if ( tController->cw_unit[ tUnit ] )
        {
            write_hex( port, (char *)"Updateing unit", (int)tController->cw_unit[ tUnit ] );
            cmd_update( tController->cw_unit[ tUnit ], ioreq );
        }
}

int cmd_execute(struct cwDevUnit *unit,struct IORequest *ioreq)
{
    int i,func,error;
    struct IOExtTD  *disk = (struct IOExtTD *) ioreq;
    struct Task     *task;
    struct MsgPort  *port;

    write_txt( unit->driver->debug_port, "Execute!!!!\n" );

    if ( disk->iotd_Req.io_Command == CMD_WRITE )
    {
//        IExec->DebugPrintF( "Not writing.\n" );
//        return 0;
    }
    if ( disk->iotd_Req.io_Command == CMD_UPDATE )
    {
#if defined(AROS)
#else
        IExec->DebugPrintF( "Not Updating.\n" );
#endif
        unit->needsUpdate = 1;
        return 0;
    }
    if ( disk->iotd_Req.io_Command == TD_MOTOR )
    {
//        write_txt( unit->driver->debug_port, "Ignoring motor instruction." );
//        return 0;
    }

    disk = (struct IOExtTD *) ioreq;
    func = -1;
    
    port = unit -> driver ->debug_port;

    for (i=0;dev_cmd_list[i].func!=NULL;i++)
    {
        if ( dev_cmd_list[i].cmd == disk -> iotd_Req.io_Command )
        {
            func = i;
            break;
        }
    }

    write_hex(port,(char *)"Command",disk -> iotd_Req.io_Command );


    if (func>-1)
    {
        write_txt(port,dev_cmd_list[i].txt);
        write_txt(port,(char *)"\n");

#if defined(AROS)
#else
        IDOS = unit -> driver -> I_DOS;
#endif

        error = dev_cmd_list[func].func( unit,(struct IOExtTD *) ioreq);

        if (error)
        {
            write_num(port,(char *)"Error",error);
            write_num(port,(char *)"Offset",disk -> iotd_Req.io_Offset);
            write_num(port,(char *)"Len",disk -> iotd_Req.io_Length);
            write_num(port,(char *)"Block",disk -> iotd_Req.io_Offset / cw_formats[ unit -> format ].blksize );
        }

        return error;
    }   
    else
    {
        write_hex( port, (char *)"Command not supported:", disk->iotd_Req.io_Command );
        return IOERR_NOCMD;
    }
}
