
/*
 * cwformat - formats disks in Catweasel drives
 *
 * Copyright (C) 1998-2002 Michael Krause
 *
 * Based on fdformat.c by Werner Almesberger (almesber@nessie.cs.id.ethz.ch)
 * (from util-linux-2.6/disk-utils)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>

#include <linux/fs.h> /* for the MAJOR macro */

#include "cwfloppy.h"

static int fd;
static struct cwfloppy_format format;

static int format_disk()
{
    int i, h = format.heads, c = format.cylinders;
    int ts = format.max_secs_per_track * format.blksize;
    void *buf = malloc(ts);

    printf("Formatting... ");
    fflush(stdout);

    if(ioctl(fd, CWFMTBEG, NULL) < 0) {
	perror("\nioctl(CWFMTBEG)");
	return 0;
    }

    for(i=0; i < h*c; i++) {
      	printf("%3d/%1d\b\b\b\b\b", i/h, i%h);
	fflush(stdout);
	if(ioctl(fd, CWFMTTRK, i) < 0) {
	    perror("\nioctl(CWFMTTRK)");
	    return 0;
	}
    }

    if(ioctl(fd, CWFMTEND, NULL) < 0) {
	perror("\nioctl(CWFMTEND)");
	return 0;
    }

    printf("done.\n");

    printf("Verifying... ");
    fflush(stdout);

    lseek(fd, 0, SEEK_SET);
    for(i=0; i < h*c; i++) {
      	printf("%3d/%1d\b\b\b\b\b", i/h, i%h);
	fflush(stdout);
	if(read(fd, buf, ts) != ts) {
	    fprintf(stderr, "\n?READ  ERROR.\n", i/h, i%h);
	    return 0;
	}
    }

    printf("done.\n");
    return 1;
}

int main(int argc, char **argv)
{
    struct stat st;

    if(argc != 2) {
	fprintf(stderr, "Usage: %s cwdev\n", argv[0]);
	return 1;
    }

    if(stat(argv[1], &st) < 0) {
	perror(argv[1]);
	return 1;
    }

    if(!S_ISBLK(st.st_mode) || MAJOR(st.st_rdev) != CWFLOPPY_MAJOR) {
	fprintf(stderr, "%s: not a cwfloppy device\n", argv[1]);
	return 1;
    }

    if((fd = open(argv[1], O_RDWR)) < 0) {
	perror(argv[1]);
	return 1;
    }
       
    if(ioctl(fd, CWGETPRM, (int)&format) < 0) {
	fprintf(stderr, "Could not determine current format type\n");
	return 1;
    }

    printf("%s\n"
	   "%d heads, %d cylinders, %d bytes per sector,\n"
	   "%d sectors per track, %d sectors total.\n\n",
	   format.name,
	   format.heads, format.cylinders, format.blksize,
	   format.max_secs_per_track, format.numblocks);

    return !format_disk();
}
