
/*
 * Catweasel -- Advanced Floppy Controller
 * Linux device driver
 * MS-DOS codec
 *
 * Copyright (C) 1998-2002 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef amigaos
#include <asm/io.h>
#endif

#include "cwfloppy.h"
#include "encoder-common.h"

static int encode_tmp[2];

void (*putfunc)(struct PCIDevice *, const unsigned char *,int ,int );

struct dos_header {
    unsigned char track, side, sec, len_desc;
    unsigned short crc;
    unsigned char gap1[22];
};

static int mfmtmp[2];

static const char msdos_sync[] = {
    2+2,1+2,2+2,1+2, 0+2,2+2,1+2,2+2,1+2, 0+2,2+2,1+2,2+2,1+2, -1
};
static const char *mfmsynccmp;

static unsigned char decode_mfm_tab[0x56], decode_mfm_tab4[0x56];

/* The CRC routines have been borrowed from drivers/block/amiflop.c */

static unsigned short dos_crc(void * data_a3, int data_d0, int data_d1, int data_d3)
{
    static unsigned char CRCTable1[] = {
    0x00,0x10,0x20,0x30,0x40,0x50,0x60,0x70,
    0x81,0x91,0xa1,0xb1,0xc1,0xd1,0xe1,0xf1,
    0x12,0x02,0x32,0x22,0x52,0x42,0x72,0x62,
    0x93,0x83,0xb3,0xa3,0xd3,0xc3,0xf3,0xe3,
    0x24,0x34,0x04,0x14,0x64,0x74,0x44,0x54,
    0xa5,0xb5,0x85,0x95,0xe5,0xf5,0xc5,0xd5,
    0x36,0x26,0x16,0x06,0x76,0x66,0x56,0x46,
    0xb7,0xa7,0x97,0x87,0xf7,0xe7,0xd7,0xc7,
    0x48,0x58,0x68,0x78,0x08,0x18,0x28,0x38,
    0xc9,0xd9,0xe9,0xf9,0x89,0x99,0xa9,0xb9,
    0x5a,0x4a,0x7a,0x6a,0x1a,0x0a,0x3a,0x2a,
    0xdb,0xcb,0xfb,0xeb,0x9b,0x8b,0xbb,0xab,
    0x6c,0x7c,0x4c,0x5c,0x2c,0x3c,0x0c,0x1c,
    0xed,0xfd,0xcd,0xdd,0xad,0xbd,0x8d,0x9d,
    0x7e,0x6e,0x5e,0x4e,0x3e,0x2e,0x1e,0x0e,
    0xff,0xef,0xdf,0xcf,0xbf,0xaf,0x9f,0x8f,
    0x91,0x81,0xb1,0xa1,0xd1,0xc1,0xf1,0xe1,
    0x10,0x00,0x30,0x20,0x50,0x40,0x70,0x60,
    0x83,0x93,0xa3,0xb3,0xc3,0xd3,0xe3,0xf3,
    0x02,0x12,0x22,0x32,0x42,0x52,0x62,0x72,
    0xb5,0xa5,0x95,0x85,0xf5,0xe5,0xd5,0xc5,
    0x34,0x24,0x14,0x04,0x74,0x64,0x54,0x44,
    0xa7,0xb7,0x87,0x97,0xe7,0xf7,0xc7,0xd7,
    0x26,0x36,0x06,0x16,0x66,0x76,0x46,0x56,
    0xd9,0xc9,0xf9,0xe9,0x99,0x89,0xb9,0xa9,
    0x58,0x48,0x78,0x68,0x18,0x08,0x38,0x28,
    0xcb,0xdb,0xeb,0xfb,0x8b,0x9b,0xab,0xbb,
    0x4a,0x5a,0x6a,0x7a,0x0a,0x1a,0x2a,0x3a,
    0xfd,0xed,0xdd,0xcd,0xbd,0xad,0x9d,0x8d,
    0x7c,0x6c,0x5c,0x4c,0x3c,0x2c,0x1c,0x0c,
    0xef,0xff,0xcf,0xdf,0xaf,0xbf,0x8f,0x9f,
    0x6e,0x7e,0x4e,0x5e,0x2e,0x3e,0x0e,0x1e
    };

    static unsigned char CRCTable2[] = {
    0x00,0x21,0x42,0x63,0x84,0xa5,0xc6,0xe7,
    0x08,0x29,0x4a,0x6b,0x8c,0xad,0xce,0xef,
    0x31,0x10,0x73,0x52,0xb5,0x94,0xf7,0xd6,
    0x39,0x18,0x7b,0x5a,0xbd,0x9c,0xff,0xde,
    0x62,0x43,0x20,0x01,0xe6,0xc7,0xa4,0x85,
    0x6a,0x4b,0x28,0x09,0xee,0xcf,0xac,0x8d,
    0x53,0x72,0x11,0x30,0xd7,0xf6,0x95,0xb4,
    0x5b,0x7a,0x19,0x38,0xdf,0xfe,0x9d,0xbc,
    0xc4,0xe5,0x86,0xa7,0x40,0x61,0x02,0x23,
    0xcc,0xed,0x8e,0xaf,0x48,0x69,0x0a,0x2b,
    0xf5,0xd4,0xb7,0x96,0x71,0x50,0x33,0x12,
    0xfd,0xdc,0xbf,0x9e,0x79,0x58,0x3b,0x1a,
    0xa6,0x87,0xe4,0xc5,0x22,0x03,0x60,0x41,
    0xae,0x8f,0xec,0xcd,0x2a,0x0b,0x68,0x49,
    0x97,0xb6,0xd5,0xf4,0x13,0x32,0x51,0x70,
    0x9f,0xbe,0xdd,0xfc,0x1b,0x3a,0x59,0x78,
    0x88,0xa9,0xca,0xeb,0x0c,0x2d,0x4e,0x6f,
    0x80,0xa1,0xc2,0xe3,0x04,0x25,0x46,0x67,
    0xb9,0x98,0xfb,0xda,0x3d,0x1c,0x7f,0x5e,
    0xb1,0x90,0xf3,0xd2,0x35,0x14,0x77,0x56,
    0xea,0xcb,0xa8,0x89,0x6e,0x4f,0x2c,0x0d,
    0xe2,0xc3,0xa0,0x81,0x66,0x47,0x24,0x05,
    0xdb,0xfa,0x99,0xb8,0x5f,0x7e,0x1d,0x3c,
    0xd3,0xf2,0x91,0xb0,0x57,0x76,0x15,0x34,
    0x4c,0x6d,0x0e,0x2f,0xc8,0xe9,0x8a,0xab,
    0x44,0x65,0x06,0x27,0xc0,0xe1,0x82,0xa3,
    0x7d,0x5c,0x3f,0x1e,0xf9,0xd8,0xbb,0x9a,
    0x75,0x54,0x37,0x16,0xf1,0xd0,0xb3,0x92,
    0x2e,0x0f,0x6c,0x4d,0xaa,0x8b,0xe8,0xc9,
    0x26,0x07,0x64,0x45,0xa2,0x83,0xe0,0xc1,
    0x1f,0x3e,0x5d,0x7c,0x9b,0xba,0xd9,0xf8,
    0x17,0x36,0x55,0x74,0x93,0xb2,0xd1,0xf0
    };

    register int i;
    register unsigned char *CRCT1, *CRCT2, *data, c, crch, crcl;

    CRCT1=CRCTable1;
    CRCT2=CRCTable2;
    data=(unsigned char *)data_a3;
    crcl=data_d1;
    crch=data_d0;
    for (i=data_d3; i>=0; i--) {
    c = (*data++) ^ crch;
    crch = CRCT1[c] ^ crcl;
    crcl = CRCT2[c];
    }
//    return (crch<<8)|crcl;
return (crcl<<8)|crch;
}

static unsigned short dos_hdr_crc (struct dos_header *hdr)
{
    return dos_crc(&(hdr->track), 0xb2, 0x30, 3); /* precomputed magic */
}

static unsigned short dos_data_crc(unsigned char *data)
{
    return dos_crc(data, 0xe2, 0x95 ,511); /* precomputed magic */
}

void codec_msdos_init()
{
    int i;
    for(i=0; i<0x56; i++) {
    decode_mfm_tab[i] = ((i&0x40)>>3) + ((i&0x10)>>2)
        + ((i&0x04)>>1) + (i&0x01);
    decode_mfm_tab4[i] = decode_mfm_tab[i] << 4;
    }
}

static int readcount;

static int get_byte_and_compare_sync(struct cwDevUnit *d)
{
    struct catweasel_contr  *controller = d ->controller ;
    int                 iobase = controller -> io_mem;
    signed char         b = PCICARD_INBYTE( controller->pcicard, iobase);
    static int tCounter = 0;

//    write_txt( d->driver->debug_port, "Looking for sync..." );

    if(b < 0)
    {
        write_num( d->driver->debug_port, (char *)"reached end of buffer. Number is %d", tCounter );
        /* end of buffer */
        return 0;
    }
    tCounter++;

    readcount++;

    b = threshtab[(int)b];

    if ( b != *mfmsynccmp )
    {
        mfmsynccmp = msdos_sync;
    }
    else
    {
        mfmsynccmp++;
        if( (unsigned char) (*mfmsynccmp) == 255) {
            /* a sync mark has been found */
            mfmsynccmp = msdos_sync;
//            write_txt( d->driver->debug_port, "Found sync!" );
            return -1;
        }
    }

    /* return 2, 3 or 4 for raw MFM 01, 001 and 0001 respectively */
    return b;
}

static int msdos_decode_mfm(struct cwDevUnit *d, int numbytes, unsigned char *buf)
{
    struct catweasel_contr  *controller = d ->controller ;
    int         t1 = mfmtmp[0];
    int         t2 = mfmtmp[1];
    int         iobase = controller -> io_mem;
    int         b, w;
    
    while(numbytes)
    {
        if( ( b=get_byte_and_compare_sync(d) ) <= 0 )
        {
            write_hex( d->driver->debug_port, (char *)"decoding error", b );
            return b;
        }

        t1 = (t1 << b) + 1;
        t2 += b;
    
        if(t2 >= 16)
        {
            t2 -= 16;
            w = t1 >> t2;
            *buf++ = decode_mfm_tab[w & 0x55] | decode_mfm_tab4[(w>>8) & 0x55];
            numbytes--;
        }
    }
    
    mfmtmp[0] = t1;
    mfmtmp[1] = t2;
    return 1;
}

int msdos_decode(struct cwDevUnit *d, int tryNumber)
{
    static const unsigned char msdos_thresholds[] = { 0x20, 0x30 };
    struct MsgPort  *port   =   d->driver->debug_port;
    int             iobase  =   d->controller-> io_mem;
    int             side        =   d->side;
    const unsigned int maxsecs = cw_formats[d->format].max_secs_per_track;
    int i;

    unsigned int todo;
    int expsec = -1;
    unsigned char marker;
    struct dos_header hdr;
    unsigned short crc;
    unsigned char *dest;

    write_num_no_enter(port, (char *)"Decoding Track", d->track);


#if 0
    struct catweasel_contr  *controller = d -> controller; 
    unsigned char bla[8],*blap = bla;
    *blap++ = controller -> PCICARD_INBYTE( pcicard, iobase);
    *blap++ = controller -> PCICARD_INBYTE( pcicard, iobase);
    *blap++ = controller -> PCICARD_INBYTE( pcicard, iobase);
    *blap++ = controller -> PCICARD_INBYTE( pcicard, iobase);
    *blap++ = controller -> PCICARD_INBYTE( pcicard, iobase);
    *blap++ = controller -> PCICARD_INBYTE( pcicard, iobase);
    *blap++ = controller -> PCICARD_INBYTE( pcicard, iobase);
    *blap++ = controller -> PCICARD_INBYTE( pcicard, iobase);
//  write_txt(port, "%02x%02x%02x%02x%02x%02x%02x%02x\n", bla[0], bla[1], bla[2], bla[3], bla[4], bla[5], bla[6], bla[7]);
    for (i=0;i<8;i++)
    {
        write_hex(d->driver->debug_port,"",bla[i]);
    }

#endif

    if(maxsecs > 32)
    {
        write_txt(port, (char *)"maxsecs > 32 not supported");
        return 0;
    }

    readcount = 0;

    todo = (1<<maxsecs) - 1; /* bit mask of decoded sectors */
    mfmsynccmp = msdos_sync;
    codec_init_threshtab(tryNumber, msdos_thresholds);

repeat:

    switch(get_byte_and_compare_sync(d))
    {
        case -1:    break;
        case 0: goto out;
        default:    goto repeat;
    }

resync:

    mfmtmp[0] = mfmtmp[1] = 0;

    if(!msdos_decode_mfm(d, 1, &marker))
        goto out;

    switch(marker)
    {
        case 0xfe: /* sector header */

            switch(msdos_decode_mfm(d, 6, (unsigned char*)&hdr))
            {
                case -1:    goto resync;
                case 0: goto out;
            }

            crc = dos_hdr_crc( &hdr );
            SwitchEndian( &crc );
            
            if(crc != hdr.crc)
            {
                write_txt(port,(char *)"header CRC mismatch\n");
                write_hex(port,(char *)"", crc );
                write_hex(port,(char *)"", hdr.crc );
                return 0;
            }
            write_num( port, (char *)"header track is", hdr.track );
            if(hdr.track != d->track)
            {
                write_txt(port,(char *)"hdr.track mismatch\n");
                write_num(port,(char *)"", hdr.track);
                return 0;
            }
            if(hdr.side != side)
            {
                write_txt(port,(char *)"hdr.side mismatch\n"); 
                write_num(port,(char *)"", hdr.side);
                write_num(port,(char *)"", side );
                return 0;
            }
            expsec = hdr.sec - 1;
            if(expsec < 0 || expsec >= maxsecs)
            {
                write_txt(port, (char *)"hdr.sec (%d) out of range (1-%d)\n");
                write_num(port,(char *)"",hdr.sec);
                write_num(port,(char *)"",maxsecs);
                return 0;
            }
            if(hdr.len_desc != 2)
            {
                write_txt(port,(char *)"unknown sector len descriptor %d\n");
                write_num(port,(char *)"",hdr.len_desc);
                return 0;
            }
            break;
    
        case 0xfb: /* sector data */

            /* we didn't have the sector header, so skip. */

            if(expsec == -1) break;

            if(!(todo & (1<<expsec)))
            {
                /* we already have this particular sector */
                expsec = -1;
                break;
            }

            dest = d->trackbuffer + expsec * 512;

            switch(msdos_decode_mfm(d, 512, dest))
            {
                case -1:    goto resync;
                case 0: goto out;
            }
    
            switch(msdos_decode_mfm(d, 2, (unsigned char*)&hdr.crc))
            {
                case -1:    goto resync;
                case 0: goto out;
            }

            crc = dos_data_crc(dest);
            SwitchEndian( &crc );

            if(crc != hdr.crc)
            {
                write_txt(port,(char *)"data CRC mismatch \n");
                write_hex(port,(char *)"",crc);
                write_hex(port,(char *)"",hdr.crc);
                return 0;
            }

            todo &= ~(1<<expsec);
            if(!todo)
            {
                return 1;
            }
            expsec = -1;
            break;
    
        default:
            write_num(port,(char *)"unknown data marker",marker);
            return 0;
    }

    goto repeat;

out:
    write_txt(port, todo == (1<<maxsecs) - 1 ? (char *)"can't find sync\n" : (char *)"track buffer incomplete\n");
    return 0;
}

/* This is IMHO a very nice routine. It converts an unencoded byte
   stream into the Catweasel RAM representation, without actually
   producing an intermediate MFM stream. */

static void
msdos_encode_mfm (struct cwDevUnit *d,
          unsigned char *data,
          int num,
          const unsigned char *mfmenc)
{
    int             prevdatabit, nextdatabit = encode_tmp[0], nullcnt = encode_tmp[1];
    unsigned char       x = 0;
    int             iobase = d -> controller -> io_mem;
    struct PCIDevice    *pcicard = d -> controller -> pcicard;

    num *= 8;

    while(num) {
    if(!(num-- & 7))
        x = *data++;
    else
        x <<= 1;

    prevdatabit = nextdatabit;
    nextdatabit = x >> 7;

    if(prevdatabit == 0) {
        /* previous data bit was 0, nullcnt is thus 1 or 2 */
        if(nextdatabit == 0) {
        /* next data bit is 0 -> clock bit is 1 -> write 01 or 001 and
           remember one null. */
        PUT(nullcnt-1);
        nullcnt = 1;
        } else {
        /* next data bit is 1 -> clock bit is 0 -> write 001 or 0001
           and clear the null counter. */
        PUT(nullcnt);
        nullcnt = 0;
        }
    } else {
        /* previous data bit was 1, nullcnt is thus 0. */
        if(nextdatabit == 0) {
        /* next data bit 0 -> clock bit also 0 -> 2 nulls in a row */
        nullcnt = 2;
        } else {
        /* next data bit 1 -> clock bit is 0 -> write "01" and
           leave nullcnt = 0. */
        PUT(0);
        }
    }
    }

    encode_tmp[0] = nextdatabit;
    encode_tmp[1] = nullcnt;
}

/*
  Some of this stuff (all the magic numbers) have been stolen
  from linux/drivers/block/amiflop.c
*/

static inline void msdos_encode_sector(struct cwDevUnit *d, int track, int sec, int side, unsigned char *data, const unsigned char *mfmenc)
{
    static struct dos_header hdr={0,0,0,2,0,
         {78,78,78,78,78,78,78,78,78,78,78,78,78,78,78,78,78,78,78,78,78,78}};
    int                 i;
    static unsigned short   crc[2]={0,0x4e4e};
    struct catweasel_contr  *controller = d -> controller;
    int                 iobase = controller -> io_mem;
    struct PCIDevice        *pcicard = controller -> pcicard;
    int                 fd = d -> driver -> fd;

    /* word before this is 0x9254, already in Catweasel RAM */
    /* the first 1 of the next data has already been written */
    /* write 6*0xaaaaaaaa, then 3*0x4489, then 0x5554 (header mark 0xfe) */
    /* 0xaaaa --> 1010101010... */
    /* 0x4489 --> 0100010010001001 */
    /* 0x5554 --> 0101010101010100 */
    for(i=0; i<6*16-1; i++)
    PUT(0);
    PUT(1);
    for(i=0; i<3; i++) {
    PUT(2); PUT(1); PUT(2); PUT(1); PUT(0);
    }
    for(i=0; i<6; i++)
    PUT(0);

    encode_tmp[0] = 0; encode_tmp[1] = 2;

    hdr.track = track;
    hdr.side = side;
    hdr.sec = sec;
    hdr.crc = dos_hdr_crc(&hdr);
    SwitchEndian( &hdr.crc );
    msdos_encode_mfm(d, (unsigned char*)&hdr, 28, mfmenc);

    /* byte before is 0x78 = 01111000 -> mfm -> x0x1x10101001010 */
    /* write 6*0xaaaaaaaa, then 3*0x4489, then 0x5545 (data mark 0xfb) */
    /* 0xaaaa --> 1010101010... */
    /* 0x4489 --> 0100010010001001 */
    /* 0x5545 --> 0101010101000101 */
    PUT(0);
    for(i=0; i<6*16-1; i++)
    PUT(0);
    PUT(1);
    for(i=0; i<3; i++) {
    PUT(2); PUT(1); PUT(2); PUT(1); PUT(0);
    }
    PUT(0); PUT(0); PUT(0); PUT(0); PUT(2); PUT(0);

    encode_tmp[0] = 1;

    msdos_encode_mfm(d, data, 512, mfmenc);

    crc[0] = dos_data_crc(data);
    SwitchEndian( &crc[0] );
    msdos_encode_mfm(d, (unsigned char*)crc, 4, mfmenc);

    /* byte before was 0x4e = 01001110 -> mfm -> x0x1x0x0x1x10100 */
    /* now write 38 * 0x92549254 */
    /* 0x9254 -> 1001001001010100 */

    PUT(1);

    for(i=0; i<2*38; i++) { PUT(1); PUT(1); PUT(1); PUT(0); PUT(0); PUT(1); }
}

/* Encoding: MFM encoding results in three typical bit vectors, that
   is 10, 100 and 1000 - or, you can also see it as 01, 001 and 0001.
   I use the latter encoding (leading zeroes), and I must take care
   not to leave out a bit by accident...   */

void msdos_encode(struct cwDevUnit *d)
{
    static const unsigned char mfmenc[3] = { 127-0x1a, 127-0x29, 127-0x37 };
    const int           iobase  = d->controller->io_mem;
    struct PCIDevice    *pcicard    = d->controller -> pcicard;
    int             side        = d->side;
    unsigned char       *data   = d->trackbuffer;
    const int           maxsecs = cw_formats[d->format].max_secs_per_track;
    struct MsgPort  *port   = d->driver->debug_port;
    int             i;

    putfunc = d->track >= 40 ? put_precomp : put_normal;

    for(i = 50; i; i--)
    {
        /* write 0x9254 -> 1001001001010100, the first 1 is left out because
        we do the 01, 001, 0001 encoding (see above) */
        PUT(1); PUT(1); PUT(1); PUT(0); PUT(0); PUT(1);
    }

    /* we have already written the first 1 of the next data */
    /* write 6* 0xaaaaaaaa, then 0x52245224, 0x52245552 */
    /* 0xaaaaaaaa -> 101010101010.... */
    /* 0x5224 -> 0101001000100100 */
    /* 0x5552 -> 0101010101010010 */

    for(i=0; i<6*16-1; i++) PUT(0);

    PUT(1);

    for(i=0; i<3; i++) {    PUT(0); PUT(1); PUT(2); PUT(1); PUT(2); }

    for(i=0; i<5; i++) PUT(0);

    PUT(1);
    PUT(0);

    for(i=0; i<2*20; i++)
    {
        /* write 0x9254 -> 1001001001010100 */
        PUT(1); PUT(1); PUT(1); PUT(0); PUT(0); PUT(1);
    }

    /* the first 1 of the following data has already been written */
    for(i=1; i<=maxsecs; i++,data+=512)
        msdos_encode_sector(d, d->track, i, side, data, mfmenc);

    /* write 0xaaa8 -> 1010101010101000, word before is always 0x9254 */
    for(i=0; i<6; i++) PUT(0);

    PUT(2);

    for(i=(2*(20+1000+(maxsecs==18)*473)); i; i--)
    {
        /* write 0x9254 -> 1001001001010100, the first 1 is left out because
        we do the 01, 001, 0001 encoding (see above) */
        PUT(1); PUT(1); PUT(1); PUT(0); PUT(0); PUT(1);
    }

    PUT(2); /* flush precomp pipeline */
}



