
/*
 * Catweasel -- Advanced Floppy Controller
 * Linux device driver
 * Amiga codec
 *
 * Copyright (C) 1998-2002 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef amigaos
#include <asm/io.h>
#endif

#include "cwfloppy.h"
#include "encoder-common.h"

struct amiga_header {
    unsigned char magic;
    unsigned char track;
    unsigned char sect;
    unsigned char ord;
    unsigned char labels[16];
    unsigned long hdrchk;
    unsigned long datachk;
};

/*
static int encode_tmp[2];
static int mfmtmp[2];


void (*putfunc)(struct PCIDevice *, const unsigned char *,int ,int );
*/
static int amiga_decode_mfm(struct cwDevUnit *d, int numbytes, void *buf)
{
    const int           iobase = d->controller->io_mem;
    struct PCIDevice    *pcicard = d -> controller -> pcicard;
    char                b;
    int             c = 0;
    unsigned char       *sbuf = (unsigned char *)buf;
    unsigned short      word;
    int             tt1 = mfmtmp[0], tt2 = mfmtmp[1];

//  printf("Amiga decode mfm\n");

//  IExec -> Disable();


    while(c < numbytes && (b = PCICARD_INBYTE( pcicard, iobase)) & 127) {
    b = threshtab[(int)b];
    tt1 = (tt1 << b) + 1;
    tt2 += b;
    
    if(tt2 >= 16) {
        tt2 -= 16;
        word = (tt1 >> tt2) & 0x5555;

        if(c < numbytes/2) {
        *sbuf++ = (word >> 8) << 1;
        *sbuf++ = (word & 0xff) << 1;
        } else {
        *sbuf++ |= (word >> 8);
        *sbuf++ |= (word & 0xff);
        }
        
        c++;
        if(c == numbytes / 2)
        sbuf = (unsigned char *)buf;
    }
    }


//  IExec -> Enable();
    
    mfmtmp[0] = tt1;
    mfmtmp[1] = tt2;
    return c == numbytes;
}

/* Who needs a real CRC anyway? :)*/

static unsigned long checksum(unsigned long *addr, int len)
{
    unsigned long csum = 0;
 
    len /= sizeof(*addr);
    while (len-- > 0)
    csum ^= *addr++;
    csum = ((csum>>1) & 0x55555555) ^ (csum & 0x55555555);
    
    return csum;
}

static int find_sync(struct cwDevUnit *d)
{
    static const char   sync[] = {  2+2,1+2,2+2,1+2, 0+2,2+2,1+2,2+2,1+2, -1    };
    const int           iobase = d->controller->io_mem;
    struct PCIDevice    *pcicard = d -> controller -> pcicard;
    const char      *synccmp = sync;
    char                b;

    while( (signed char) (b = PCICARD_INBYTE( pcicard, iobase)) >= 0) {
    if(threshtab[(int)b] != *synccmp) {
        synccmp = sync;
    } else {
        synccmp++;
        if( (signed char) (*synccmp) == -1)
        return 1;
    }
    }

    return 0;
}

int amiga_decode(struct cwDevUnit *d, int tryNumber)
{
    static const unsigned char amiga_thresholds[] = { 0x22, 0x30 }; // 27, 38 for 5.25"
    const int           iobase = d->controller->io_mem;
    const unsigned int  maxsecs = cw_formats[d->format].max_secs_per_track;
    struct PCIDevice    *pcicard = d -> controller -> pcicard;
    struct MsgPort  *port = d->driver->debug_port;
    int             track = d -> track;
    int             side = d -> side;

    int sec_count = 0;

    int todo;
    struct amiga_header hdr;
    unsigned char *dest;
    unsigned long crc;

    todo = (1<<maxsecs) - 1; /* bit mask of decoded sectors */
    codec_init_threshtab(tryNumber, amiga_thresholds);

  repeat: /* oh, this is sooo evil. */

    if(!find_sync(d))
    {
        write_txt( port, "No sync marker found.\n" );
        goto out;
    }

    mfmtmp[0] = mfmtmp[1] = 0;

    if(!amiga_decode_mfm(d, 4, &hdr.magic))
        goto out;
    if(!amiga_decode_mfm(d, 16, &hdr.labels))
        goto out;
    if(!amiga_decode_mfm(d, 4, &hdr.hdrchk))
        goto out;
    if(!amiga_decode_mfm(d, 4, &hdr.datachk))
        goto out;
    
    crc = checksum((unsigned long *)&hdr, 20);
    if(hdr.hdrchk != crc) 
    {
        write_hex(port,(char *)"  header checksum mismatch",hdr.hdrchk);
        write_hex(port,(char *)"  crc",crc);
        return 0;
    }
    
    if(hdr.magic != 0xff) 
    {
        return 0;
    }
    
    if(hdr.track != 2*d->track+side) {
    return 0;
    }
    
    if(hdr.sect >= maxsecs) {
//  write_txt(fd, "hdr.sect"); 
//  write_num(fd,hdr.sect);
//  write_txt(fd, "out of range");
//  write_num(fd,maxsecs-1);
    return 0;
    }
/*
    write_num_no_enter(port,(char *)"  Track",hdr.track);
    write_num(  port,(char *)"  Sector",hdr.sect);
*/
    if(!(todo & (1<<hdr.sect)))
    goto repeat; /* we already have this sector */

    dest = d->trackbuffer + hdr.sect * 512;

    if(!amiga_decode_mfm(d, 512, dest))
    {
        write_txt( port, "Error decoding MFM data.\n" );
        goto out;
    }
    crc = checksum((unsigned long *)dest, 512);
    if(hdr.datachk != crc) {
//  write_txt(fd, "data checksum mismatch");
//  write_hex(fd,hdr.datachk);
//  write_hex(fd, crc);
    return 0;
    }

    sec_count ++;

    todo &= ~(1<<hdr.sect);
    if(!todo)
    return 1;

    goto repeat;

out:
//  write_txt(      port, todo == (1<<maxsecs) - 1 ?   "AMIGA: can't find sync\n" : "AMIGA: track buffer incomplete\n");
    write_num(  port,(char *)"sector count",sec_count );
    return 0;
}

/* The Amiga does MFM encoding for the even bits and odd bits
   separately. This has the advantage that the en-/decoding (which, on
   the Amiga hardware, is not done by the floppy controller itself)
   can be done efficiently using the CPU (or the blitter) without
   having to shuffle bits around too much.

   This routine does the encoding without any intermediate buffer. It
   may well be possible to find a faster algorithm. */

static void amiga_encode_mfm (struct cwDevUnit *d,
          const unsigned char *data,
          int num,
          const unsigned char *mfmenc,
          int shift)
{
    int prevdatabit, nextdatabit = encode_tmp[0], nullcnt = encode_tmp[1];
    unsigned char       x = 0;
    int             iobase = d -> controller -> io_mem;
    struct PCIDevice    *pcicard = d -> controller -> pcicard;


    num *= 4;

    while(num) {
    if(!(num-- & 3))
        x = (*data++) << shift;
    else
        x <<= 2;

    prevdatabit = nextdatabit;
    nextdatabit = x >> 7;

    if(prevdatabit == 0) {
        /* previous data bit was 0, nullcnt is thus 1 or 2 */
        if(nextdatabit == 0) {
        /* next data bit is 0 -> clock bit is 1 -> write 01 or 001 and
           remember one null. */
        PUT(nullcnt-1);
        nullcnt = 1;
        } else {
        /* next data bit is 1 -> clock bit is 0 -> write 001 or 0001
           and clear the null counter. */
        PUT(nullcnt);
        nullcnt = 0;
        }
    } else {
        /* previous data bit was 1, nullcnt is thus 0. */
        if(nextdatabit == 0) {
        /* next data bit 0 -> clock bit also 0 -> 2 nulls in a row */
        nullcnt = 2;
        } else {
        /* next data bit 1 -> clock bit is 0 -> write "01" and
           leave nullcnt = 0. */
        PUT(0);
        }
    }
    }

    encode_tmp[0] = nextdatabit;
    encode_tmp[1] = nullcnt;
}

static inline void amiga_encode_sector(struct cwDevUnit *d, int track, int sec, int maxsecs, unsigned char *data, const unsigned char *mfmenc)
{
    static const unsigned char  mark1[] = { 0, 0, 0, 0 };
    struct PCIDevice            *pcicard = d -> controller -> pcicard;
    int                     iobase = d -> controller -> io_mem;
    int                     i;
    struct amiga_header     hdr;
    
    /* intro mark */
    amiga_encode_mfm(d, mark1, 4, mfmenc, 0);

    /* write two syncs 0x4489 */
    PUT(1); PUT(2); PUT(1); PUT(2); PUT(1);
    PUT(0); PUT(2); PUT(1); PUT(2); PUT(1);

    hdr.magic = 0xFF;
    hdr.track = track;
    hdr.sect = sec;
    hdr.ord = maxsecs - sec;

    for (i = 0; i < 16; i++)
    hdr.labels[i] = 0;

    hdr.hdrchk = checksum((unsigned long *)&hdr,
              (char *)&hdr.hdrchk-(char *)&hdr);
    hdr.datachk = checksum((unsigned long *)data, 512);

    encode_tmp[1] = 0; encode_tmp[0] = 1;
    amiga_encode_mfm(d, &hdr.magic, 4, mfmenc, 0);
    amiga_encode_mfm(d, &hdr.magic, 4, mfmenc, 1);
    amiga_encode_mfm(d, (unsigned char*)&hdr.labels, 16, mfmenc, 0);
    amiga_encode_mfm(d, (unsigned char*)&hdr.labels, 16, mfmenc, 1);
    amiga_encode_mfm(d, (unsigned char*)&hdr.hdrchk, 4, mfmenc, 0);
    amiga_encode_mfm(d, (unsigned char*)&hdr.hdrchk, 4, mfmenc, 1);
    amiga_encode_mfm(d, (unsigned char*)&hdr.datachk, 4, mfmenc, 0);
    amiga_encode_mfm(d, (unsigned char*)&hdr.datachk, 4, mfmenc, 1);
    amiga_encode_mfm(d, data, 512, mfmenc, 0);
    amiga_encode_mfm(d, data, 512, mfmenc, 1);
}

void amiga_encode(struct cwDevUnit *d)
{
    const int           iobase = d->controller->io_mem;
    const unsigned int  maxsecs = cw_formats[d->format].max_secs_per_track;
    struct PCIDevice    *pcicard = d -> controller -> pcicard;
    int             i;
    unsigned char       *data = d->trackbuffer;
    int             fd = d->driver->fd;
    int             side = d -> side;

    static const unsigned char enddata[] = { 0, 0 };
    static const unsigned char mfmenc[3] = { 0x80-0x1b, 0x80-0x2a, 0x80-0x38 };

    putfunc = d->track >= 40 && maxsecs == 22 ? put_precomp : put_normal;

    /* gap space */
    for(i = 0; i < 16 * 500 * (1+(maxsecs > 11)); i++) {
    PUT(0);
    }

    encode_tmp[0] = 0;
    encode_tmp[1] = 1;

//  write_txt(fd,"The sector, data addr");
//  write_hex(fd, (unsigned int) data);

    /* the sectors */
    for(i = 0; i<maxsecs; i++,data+=512)
    amiga_encode_sector(d, 2*d->track +side, i, maxsecs,
                data, mfmenc);

    /* end mark */
    amiga_encode_mfm(d, enddata, 2, mfmenc, 0);
}
