
/*
 * Catweasel -- Advanced Floppy Controller
 * Linux device driver
 * Low-level routines (header)
 *
 * Copyright (C) 1998-2002 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef _CATWEASEL_H
#define _CATWEASEL_H

#include <exec/types.h>
#include <exec/devices.h>
#include <dos/dos.h>

#include <clib/exec_protos.h>
#if defined(AROS)
#  include <clib/alib_protos.h>
#  include "AROS_PCI.h"
#  include "dev_print.h"
//#  include "catweasel_libdefs.h"
#endif

#define NUMBER_OF_SUPPORTED_FORMATS 4

#include "cwfloppy.h"

struct cw_driver;
struct catweasel_drive;
struct catweasel_contr;

struct cwDevUnit {
    struct Unit         real_unit;
    UWORD               cw_unit_num;
    UWORD               *pad1;
    struct cw_driver        *driver;
    struct DriveGeometry    *drive_geo;
    struct catweasel_contr  *controller;
    struct Interrupt        *interrupt;

    unsigned char                    *buffered[200]; // keep track of bufferd tracks
    char                    update[200];
    unsigned char                    *trackbuffer;

    int timeout;            // activity check

    int type;               // 0 = not present, 1 = 3.5" 
    int track;              // current r/w head position (0..79)
    int side;
    int diskindrive;            // 0 = no disk, 1 = disk in drive 
    int motor;          // O = off, 1= On
    int wprot;              // 0 = not, 1 = write protected 
    int format;
    int ticks;              // counter 
    int needsUpdate;        // Do we need to update this when we get a moment?
    int motorChanged;       // Does the motor need to be turned off?
};

struct cw_driver {
    struct Device           cw_device;
    UWORD               pad;
#if defined(AROS)
#else
    struct ExecIFace        *I_Exec;
#endif
    void                    *DOSBase;
#if defined(AROS)    
#else
    struct DOSIFace     *I_DOS;
#endif
    void                    *TimerBase;
    struct TimerIFace       *I_Timer;
    void                    *ExpBase;
    struct ExpansionIFace   *I_Exp;
    struct PCIIFace     *I_PCI;
    BPTR                SegList;
    UBYTE               *cw_base;
    struct catweasel_contr  *controller[6];
    int                 num_controllers;
    int                 num_units;
    struct Task         *task;
    int                 fd;
    struct MsgPort      *debug_port;
#if defined(AROS)
    struct List         mLegacyBuses;
    struct List         mProbedBuses;
    struct MinList      mBuses;
#endif
};


typedef struct catweasel_drive {
    int number;         /* Drive number: 0 or 1 */
    int type;           /* 0 = not present, 1 = 3.5" */
    int track;          /* current r/w head position (0..79) */
    int diskindrive;        /* 0 = no disk, 1 = disk in drive */
    int wprot;          /* 0 = not, 1 = write protected */
} catweasel_drive;


typedef struct catweasel_contr {
    int                     type;           /* see CATWEASEL_TYPE_* defines below */
    int                     mShutDownNow;  // Have we finished?
    struct cw_driver            *driver;
    struct PCIDevice            *pcicard;
    struct PCIResourceRange *io_range;
    struct MsgPort          *MsgPort;
#if defined(AROS)
    struct timerequest      *ms_timer;
#else    
    struct TimeRequest          *ms_timer;
#endif
    struct Task             *cw_task;
    int                     io_base;            /* 0 = not present (factory default is 0x320) */
    struct cwDevUnit            *cw_unit[ NUMBER_OF_SUPPORTED_FORMATS * 2]; // Each format needs its own structure, and there's two units.
    int                     control_register;   /* contents of control register */
    unsigned                    char crm_sel0;      /* bit masks for the control / status register */
    unsigned                    char crm_sel1;
    unsigned                    char crm_mot0;
    unsigned                    char crm_mot1;
    unsigned                    char crm_dir;
    unsigned                    char crm_step;
    unsigned                    char srm_trk0;
    unsigned                    char srm_dchg;
    unsigned                    char srm_writ;
    int                     io_sr;          /* IO port of control / status register */
    int                     io_mem;         /* IO port of memory register */
    int mDiskChanged[ 2 ];  // Has a disk been changed in each unit?
} catweasel_contr;

#if defined(AROS)
#  include "catweasel_libdefs.h"
#endif

struct ext_code_type
{
    void        (* code) (struct catweasel_contr *controller, ULONG arg1);
    ULONG   arg1;
};

extern const struct cwfloppy_format cw_formats[ 4 ];

#if defined(AROS)
void cw_udelay(struct timerequest *ior,int usec);
void cw_msdelay(struct timerequest *ior,int msec);
#else
void cw_udelay(struct TimeRequest *ior,int usec);
void cw_msdelay(struct TimeRequest *ior,int msec);
#endif
#include "encoder-common.h"


#define CATWEASEL_TYPE_NONE  -1
#define CATWEASEL_TYPE_MK1    1
#define CATWEASEL_TYPE_MK3    3
#define CATWEASEL_TYPE_MK4    4

// pci ids 
#define CW_VENDOR       0xe159
#define CW_DEVICE       0x0001
#define CW_MK3_SUBVENDOR    0x1212
#define CW_MK3_SUBDEVICE    0x0002
#define CW_MK4_SUBVENDOR1   0x5213
#define CW_MK4_SUBVENDOR2   0x5200
#define CW_MK4_SUBDEVICE1   0x0002
#define CW_MK4_SUBDEVICE2   0x0003

// generic registers 
#define CW_DATA_DIRECTION   0x02
#define CW_SELECT_BANK      0x03
#define CW_PORT_OUT_DIR     CW_SELECT_BANK
#define CW_PORT_AUX     0x05
#define CW_PORT_IN_DIR      0x07

// values for CW_SELECT_BANK 
#define CW_BANK_RESETFPGA   0x00
#define CW_BANK_FIFO        0x01
#define CW_BANK_FLOPPY      0x41
#define CW_BANK_IO      0x81
#define CW_BANK_IRQ     0xC1

// registers in FLOPPY bank 
#define CW_FLOPPY_JOYDAT    0xC0
#define CW_FLOPPY_JOYBUT    0xC8
#define CW_FLOPPY_JOYBUT_DIR    0xCC
#define CW_FLOPPY_KEYDAT    0xD0
#define CW_FLOPPY_KEYSTATUS 0xD4
#define CW_FLOPPY_SIDDAT    0xD8
#define CW_FLOPPY_SIDADDR   0xDC
#define CW_FLOPPY_MEMORY    0xE0
#define CW_FLOPPY_RESETPOINTER  0xE4
#define CW_FLOPPY_CONTROL   0xE8
#define CW_FLOPPY_OPTION    0xEC
#define CW_FLOPPY_START_A   0xF0
#define CW_FLOPPY_START_B   0xF4
#define CW_FLOPPY_IRQ       0xFC

// registers in IO bank 
#define CW_IO_MOUSEY1       0xC0
#define CW_IO_MOUSEX1       0xC4
#define CW_IO_MOUSEY2       0xC8
#define CW_IO_MOUSEX2       0xCC
#define CW_IO_BUTTON        0xD0

// registers in CW_BANK_IRQ 
#define CW_IRQ_R0       0xC0
#define CW_IRQ_R1       0xC4
#define CW_IRQ_M0       0xC8
#define CW_IRQ_M1       0xCC
// bits in registers in CW_BANK_IRQ 
#define CW_IRQ_MK3FLOPPY    0x01
#define CW_IRQ_INDEX        0x02
#define CW_IRQ_FLOPPY_START 0x04
#define CW_IRQ_FLOPPY_END   0x08
#define CW_IRQ_SID_FIFO_HALF_EMPTY  0x10
#define CW_IRQ_SID_READ     0x20
#define CW_IRQ_MEM_EQUAL    0x40
#define CW_IRQ_KEYBOARD     0x80
#define CW_IRQ_SID_FIFO_EMPTY   0x01
#define CW_IRQ_SID_FEEDBACK 0x02

/* Initialize a Catweasel controller; c->iobase and c->msdelay must have
   been initialized -- msdelay might be used */
void catweasel_init_controller(struct catweasel_contr *c);

/* Initialize a Catweasel controller; c->iobase and c->msdelay must have
   been initialized -- msdelay might be used */
void catweasel_init_unit(struct catweasel_contr *c);

// Reset the controller 
void catweasel_free_controller(struct catweasel_contr *c);

// Set current drive select mask 
void catweasel_select(struct catweasel_contr *c, int dr0, int dr1);

// Start/stop the drive's motor 
void catweasel_set_motor(struct cwDevUnit *d, int on);

// Move the r/w head -- msdelay might be used 
int catweasel_seek(struct cwDevUnit *d, int track);

/* Check for a disk change and update d->diskindrive
   -- msdelay might be used. Returns 1 == disk has been changed */
int catweasel_disk_changed(struct cwDevUnit *d);

// Check if disk in selected drive is write protected. 
int catweasel_write_protected(struct cwDevUnit *d);

// Read data -- msdelay will be used 
int catweasel_read(struct cwDevUnit *d);

// Read raw data!
int catweasel_read_raw_data( struct cwDevUnit *d, BYTE *data, int maximumLength );

// Or Write it!
int catweasel_write_raw_data( struct cwDevUnit *d, BYTE *data, int maximumLength );

/* Write data -- msdelay will be used. If time == -1, the write will
   be started at the index pulse and stopped at the next index pulse,
   or earlier if the Catweasel RAM contains a 128 end byte.  The
   function returns after the write has finished. */
int catweasel_write(struct cwDevUnit *d);

// endianness issues!
void SwitchEndian( unsigned short *value );

#if defined (AROS)
#  define PCICARD_OUTBYTE(device,r,v) PCIDevice_OutByte(device, r, v)
#  define PCICARD_INBYTE(device,r) PCIDevice_InByte(device, r)
#else
#  define PCICARD_OUTBYTE(device,r,v) device->OutByte(r, v)
#  define PCICARD_INBYTE(device,r) device->InByte(r)
#endif 

__inline__ void CWSetCReg(struct catweasel_contr *c, unsigned char clear, unsigned char set);

#if defined(AROS)
extern struct catweasel_contr *sharedController;
void task_begin();
#else            
void task_begin(  struct catweasel_contr  *contr );
#endif

#if defined(AROS)
#  define EXEC_CALL
#  define DOS_CALL
#else
#  define EXEC_CALL IExec->
#  define DOS_CALL IDOS->
#endif

#endif // _CATWEASEL_H 
