
/*
 * Catweasel -- Advanced Floppy Controller
 * Linux device driver
 * Data common to all encoders
 *
 * Copyright (C) 1998-2002 Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef amigaos
#include <asm/io.h>
#endif

#include "cwfloppy.h"

/* Precompensation is used to compensate a physical effect which
   occurs when a short magnetic pulse is written between two longer
   pulses. With MFM encoding, we have three pulse lengths; when the
   shortest pulse is located between two longer ones, we must make the
   short one even shorter (and the other ones longer in order to keep
   the overall writing length the same). In other words, a short pulse
   after a long one extends the latter (towards the short one, on the
   time axis) and the short one gets shortened even more. A short
   pulse before a long one is treated in the same way. A short pulse
   before or after a short one remains unchanged. (Alle Klarheiten
   beseitigt? :D)

   The complex reality (which I have to admit I don't really
   understand) boils down to this simple recipe (P = previous pulse
   length, C = current pulse length):

   Is C a long pulse?
    --> When P is also long: write P, P <-- C.
    --> When P is short: write P-1, P <-- C+1.
   or is C a short pulse?
    --> When P is also short: write P, P <-- C.
    --> When P is long: write P+1, P <-- C-1.

   Manually trying to apply these rules to the data stream
   "021001000012020" should help understanding the system. You will
   also notice, that (oh great!) all this mess reduces to a very, very
   small routine.

   Notes: The amount of adjustment is exactly one Catweasel clock and
   that (X is a long pulse) <==> (X >= 1). A pulse is made longer by
   subtracting from the RAM value (the Catweasel counts up when
   writing).

   It's just funny that Amigas (at least my A4000) don't care whether
   the Catweasel writes HD disks with precompensation or without. It
   reads them without errors in any case. The PC controller is a lot
   more picky about it though. Anyway, precompensation actually does
   magic.  It's interesting to compare the inner tracks with
   'test-histo', with and without precompensation.

*/

void put_normal(struct PCIDevice    *pcicard, const unsigned char *mfmenc,int x,int iobase)
{
    PCICARD_OUTBYTE(pcicard, iobase, mfmenc[ x ] );
}

void put_precomp(struct PCIDevice *pcicard, const unsigned char *mfmenc,int x,int iobase)
{
    static int previous = 0, adjust = 0;
    int newadjust = (previous >= 1) - (x >= 1);

    PCICARD_OUTBYTE( pcicard,  iobase, mfmenc[previous] - adjust - newadjust );
    previous = x;
    adjust = -newadjust;
}

#if ENCODE_CALC_WRITETIME
void calc_time (struct PCIDevice *pcicard,cw_drive_struct *d)
{
    int c = 0;
    int len = 0;
    char b;
    char buf[256];

#if 0
    int i;
    char *bp;

    for(i = 0; i < 256; i++) {
    buf[i] = 0;
    }
#endif

    Update these for the MK3 catweasel;
    pcicard -> InByte(iobase+1);
    pcicard -> InByte(iobase+0);

    while((b = inb(iobase)) >= 0 && c < 1<<17) {
    buf[128+b] = 1;
    len += (127-b);
    c++;
    }

#if 0
    for(i = 0, bp = buf; i < 256 / 16; i++) {
    printk("%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d\n",
           bp[0], bp[1], bp[2], bp[3], 
           bp[4], bp[5], bp[6], bp[7], 
           bp[8], bp[9], bp[10], bp[11], 
           bp[12], bp[13], bp[14], bp[15]);
    bp += 16;
    }
#endif

    printk("%d bytes; %d cycles (28M:%d; 14M:%d)ms\n",
       c, len, 2*len/28322, 2*len/28322*2);
}
#endif

unsigned char threshtab[128];

static void codec_makethresh(int tryNumber, const unsigned char *origt, unsigned char *t, int numthresh)
{
    static unsigned char tab[10] = { 0, 0, 0, 0, -1, -2, 1, 2, -1, 1 };

    while(numthresh--)
        t[numthresh] = origt[numthresh] + tab[tryNumber];
}

void
codec_init_threshtab(int tryNumber,
             const unsigned char *origt)
{
    static unsigned char old_thresholds[2] = { 0, 0 };
    unsigned char t[2];
    int a, i;

    codec_makethresh(tryNumber, origt, t, 2);

    if(*(unsigned short*)t == *(unsigned short*)old_thresholds)
    return;

    for(i=0,a=2; i<128; i++) {
    if(i == t[0] || i == t[1])
        a++;
    threshtab[i] = a;
    }

    *(unsigned short*)&old_thresholds = *(unsigned short*)t;
}
